
//~~~~~~~~~~~~~~~~~~~~~~~~~~~
#include <gc.h>
//~~~~~~~~~~~~~~~~~~~~~~~~~~~

void RenderSine( TGC* gc, double sineRate, INT32 npeak, INT32 ppeak );
void RenderMultiBurst( TGC* gc, INT32 npeak, INT32 ppeak );

/******************************************************************************
*
*  RenderMultiBurst 
*
*******************************************************************************
*  Draws a multiburst pattern as follows.
*  Flags are 2/25 active high, 2/25 low, & 2/25 mid.
*  Burst packets (6) are 3/25 active wide.
*
*******************************************************************************
*  Usage:
*     RenderMultiBurst( TGC* gc, int32 npeak, int32 ppeak );
*
*  Arguments:
*     gc    - the graphics context being drawn on.
*     npeak - see below
*     ppeak - see below
*
*******************************************************************************
*/
void RenderMultiBurst( TGC* gc, INT32 npeak, INT32 ppeak )
{
  // Since GetVersions can return 0 and we want a positive sine rate, we add 1.
  double sineRate = (gc->GetVersions() + 1) * 500000;

  /* Draw the sine waves. */
  RenderSine( gc, sineRate, npeak, ppeak);

  /* Fill the screen. */
  gc->TranspOFF(); /* turn transparency off so black areas get filled too */
  gc->FastMode( true );
  gc->CurtainFillRect( gc->width, gc->height, 0, 0, 0 );
}


/******************************************************************************
*
*  RenderSine 
*
*******************************************************************************
*  Draws a multiburst pattern as follows.
*  Flags are 2/25 active high, 2/25 low, & 2/25 mid.
*  Burst packets (6) are 3/25 active wide.
*
*******************************************************************************
*  Usage:
*     RenderSine( TGC* gc, double sineRate, int32 npeak,
*               int32 ppeak);
*
*  Arguments:
*     gc       - the graphics context being drawn on.
*     sineRate - frequency of wave to draw. (Hz)
*     ppeak		 - positive peak (dac) value. (0-255)
*     npeak		 - negative peak (dac) value. (0-255)
*
*******************************************************************************
*/
void RenderSine( TGC* gc, double sineRate, INT32 npeak,
                       INT32 ppeak )
{
  INT32  i, n;
  double peak, samplePeriod ;
  TMath ml;

  samplePeriod = 2 * PI * (1.0 / gc->GetPixelRate() ) * sineRate; /* period in radians */
  peak = ((double)ppeak - (double)npeak) / 2.0;          /* peak amplitude. */

  for( i = 1; i < gc->width; i++ )
  {
    n = ml.Round( (1.0 + (ml.Sin((double)i * samplePeriod))) * peak) + npeak;
    gc->Transparent( gc->RepPixel( n ) );
    gc->DrawPoint( i, 0 );
  }
}


/******************************************************************************
*
*  Render_tceburst  
*
*******************************************************************************
*  Draws burst patterns of selectale frequency.
*
*******************************************************************************
*  Usage:
*     Render_tceburst( TGC* gc );
*
*  Arguments:
*     gc - the graphics context being drawn on.
*
*  Returns:
*     bool completed - true if the image could be drawn.
*                    - false if it can't be drawn.
*
*******************************************************************************
*/
bool Render_tceburst( TGC* gc )
{
  INT32 ppeak, npeak ;

  switch( gc->GetAnalogVideoType() )
  {
    case 3:
    case 4:
      break;

    default:
      break;
      return false;
  }

  gc->Versions( 66 );

  gc->ZeroActive();

  gc->SelectColor( gc->GetColorLast() );
  gc->SetLevelGray( 255 );
  gc->Transparent( gc->RepPixel( gc->GetColorLast() ) );
  gc->SetLut( BW|GRAYALL|NOGAMMA|TOBLANK );

  ppeak = 205;  /* 80 IRE */
  npeak =  51;  /* 20 IRE */
  RenderMultiBurst( gc, npeak, ppeak );

  return true;
}
